"""
Batch Image Header Watermark (Non-Destructive / Above Image)
------------------------------------------------------------

Features:
- ADDS height to the image (does not cover original pixels).
- Creates a clean black header bar above the photo.
- Auto-fits text to the header width.
- Batch processes all images in the folder.

Requirements:
    pip install pillow
"""

import os
from PIL import Image, ImageDraw, ImageFont

IMAGE_EXTS = (".png", ".jpg", ".jpeg", ".webp", ".bmp", ".tiff")

def load_font(size):
    """Try common fonts; fall back safely."""
    # List of fonts to try (Windows/Linux/Mac common paths)
    options = ["arialbd.ttf", "arial.ttf", "calibrib.ttf", "DejaVuSans-Bold.ttf"]
    for font_name in options:
        try:
            return ImageFont.truetype(font_name, size)
        except:
            continue
    return ImageFont.load_default()

def fit_font(draw, text, max_width, max_height, start_size):
    """Shrink font until text fits within the header box."""
    size = start_size
    while size > 10:
        font = load_font(size)
        bbox = draw.textbbox((0, 0), text, font=font)
        text_w = bbox[2] - bbox[0]
        text_h = bbox[3] - bbox[1]
        
        # Check if fits width AND height (with padding)
        if text_w <= max_width and text_h <= (max_height * 0.8):
            return font
        size -= 2
    return load_font(size)

def watermark_image(path, text):
    # 1. Open original
    img = Image.open(path).convert("RGBA")
    w, h = img.size

    # 2. Calculate Header Height 
    # (15% of image height, or minimum 80px, max 300px to avoid huge headers on hi-res)
    header_h = int(h * 0.15)
    header_h = max(header_h, 80)
    
    # 3. Create NEW Canvas (Width = same, Height = Original + Header)
    # Background is Solid Black
    new_h = h + header_h
    new_img = Image.new("RGBA", (w, new_h), (0, 0, 0, 255))

    # 4. Paste the Original Image UNDER the header
    new_img.paste(img, (0, header_h))

    draw = ImageDraw.Draw(new_img)

    # 5. Determine Font Size
    # Start trying at 70% of the header height
    start_size = int(header_h * 0.7)
    font = fit_font(draw, text, int(w * 0.95), header_h, start_size)

    # 6. Center Text in the Header Area
    bbox = draw.textbbox((0, 0), text, font=font)
    text_w = bbox[2] - bbox[0]
    text_h = bbox[3] - bbox[1]

    x = (w - text_w) // 2
    # Vertically center within just the header_h
    y = (header_h - text_h) // 2

    # 7. Draw Text (White text, no heavy effects needed on solid black)
    draw.text((x, y), text, font=font, fill=(255, 255, 255, 255))

    # 8. Save
    base, ext = os.path.splitext(path)
    out_path = f"{base}_header{ext}"
    
    # Convert back to RGB to save as JPG (removes alpha channel)
    new_img.convert("RGB").save(out_path, quality=95)

    print(f"✔ Added header to: {os.path.basename(out_path)}")

def main():
    text = input("Enter text for the header bar: ").strip()

    if not text:
        print("❌ No text entered.")
        return

    images = [f for f in os.listdir(".") if f.lower().endswith(IMAGE_EXTS)]
    
    # Filter out files that already have _header in name to prevent double processing
    images = [f for f in images if "_header" not in f]

    if not images:
        print("❌ No images found.")
        return

    print(f"\nProcessing {len(images)} images...\n")

    for img in images:
        try:
            watermark_image(img, text)
        except Exception as e:
            print(f"⚠ Skipped {img}: {e}")

    print("\n✅ Done. Images extended with top header.")

if __name__ == "__main__":
    main()